/**************************************************************************************
 
   Copyright (c) Hilscher GmbH. All Rights Reserved.
 
 **************************************************************************************
 
   Filename:
    $Id: TCPLayer.h 5335 2013-05-07 07:31:27Z MichaelT $
   Last Modification:
    $Author: MichaelT $
    $Date: 2013-05-07 09:31:27 +0200 (Di, 07 Mai 2013) $
    $Revision: 5335 $
   
   Targets:
     Win32/ANSI   : yes
     Win32/Unicode: no
     WinCE        : no
 
   Description:
     Physical layer for TCP interfaces

   Changes:
 
     Version   Date        Author   Description
     ----------------------------------------------------------------------------------
     2         27.11.09    SS       Review
     1         28.07.09    SS       initial version
 
**************************************************************************************/

/*****************************************************************************/
/*! \file TCPLayer.h
*   Physical layer for the TCP interfaces                                    */
/*****************************************************************************/

#pragma once

#pragma warning(push)
#pragma warning(disable: 4702)
#include <map>
#include <queue>
#pragma warning(pop)
#include <winsock2.h>
#include "connectorAPI.h"
#include "TCPConfig.h"

/* Forward declaration */
class CTCPInterface;

/*****************************************************************************/
/*! \addtogroup netX_CONNECTOR_TCP netX TCP Connector                        */
/*! \{                                                                       */
/*****************************************************************************/

///////////////////////////////////////////////////////////////////////////////////////////
/// TCP Interface definitions
///////////////////////////////////////////////////////////////////////////////////////////
#define TCP_CONNECTOR_IDENTIFIER  "TCP" /*!< Identifier of the TCP connector      */
#define TCP_CONNECTOR_DESCRIPTION "TCP Connection" /*!< Description of the TCP connector      */
#define CONNECTOR_THREAD_TIMEOUT   1000 /*!< Internal timeout for thread handling */

///////////////////////////////////////////////////////////////////////////////////////////
/// \class CTCPLayer
/// Physical Layer for TCP interfaces
///////////////////////////////////////////////////////////////////////////////////////////
class CTCPLayer
{
public:
   CTCPLayer(void);
  ~CTCPLayer(void);

  long                Init                   ( PFN_NETXCON_DEVICE_NOTIFY_CALLBACK pfnDevNotifyCallback, 
                                               void*                              pvUser);
  void                Deinit                 ( void);
  
  void                Reinit                 ( void);
  
  CTCPInterface*      CreateInterface        ( std::string szDeviceName);
  CTCPInterface*      CreateInterface        ( DWORD  dwIPAddress, 
                                               USHORT usTCPPort);

  long                ValidateInterface      ( PCONNECTOR_INTERFACE pcInterface);

  void                SetNotificationEvent   ( std::string&                  szInterfaceName, 
                                               NETX_INTERFACE_NOTIFICATION_E eNotification );

  void                LockInterfaceAccess    ( void);
  void                UnlockInterfaceAccess  ( void);

  CTCPInterface*      FindInterface          ( const char* szInterface);
  CTCPInterface*      FindInterface          ( ULONG ulIpAddress, USHORT usPort);

protected:
  ///////////////////////////////////////////////////////////////////////////////////////////
  /// Interface notify struct
  ///////////////////////////////////////////////////////////////////////////////////////////
  typedef struct INTERFACE_NOTIFY_Ttag
  {
    std::string                   szInterfaceName;   /*!< Handle to interface                      */
    NETX_INTERFACE_NOTIFICATION_E eNotification;     /*!< Interface attached or detached           */

  } INTERFACE_NOTIFY_T, *PINTERFACE_NOTIFY_T;

  ///////////////////////////////////////////////////////////////////////////////////////////
  /// Notify Queue
  ///////////////////////////////////////////////////////////////////////////////////////////
  typedef std::queue<INTERFACE_NOTIFY_T> NOTIFY_QUEUE;
  
  ///////////////////////////////////////////////////////////////////////////////////////////
  /// Interface Map
  ///////////////////////////////////////////////////////////////////////////////////////////
  typedef std::map<std::string, CTCPInterface*> INTERFACE_MAP;

  INTERFACE_MAP                       m_cmInterfaceControlMap;        /*!< All available interfaces                       */
  NOTIFY_QUEUE                        m_cqNotifyQueue;                /*!< Queue holding interface notifications          */
  PFN_NETXCON_DEVICE_NOTIFY_CALLBACK  m_pfnNotifyCallback;            /*!< Callback for the notify calls                  */
  void*                               m_pvNotifyUser;                 /*!< User pointer for the notify calls              */
   
  HANDLE                              m_hInterfaceNotifyThread;       /*!< Thread for interface control                   */
  HANDLE                              m_hInterfaceNotifyThreadStop;   /*!< Control interface thread signal stop           */
  HANDLE                              m_hInterfaceNotifyEvent;        /*!< Signals Notifcation Event of the interface     */
  CRITICAL_SECTION                    m_tcsInterfaceLock;             /*!< Critical Section for the interface map         */      
  CRITICAL_SECTION                    m_tcsNotifyQueueLock;           /*!< Critical Section for the notification queue    */      

  static DWORD WINAPI InterfaceNotifyThread ( void* pvParam);
  
  std::string         CreateInterfaceName   ( void);
};

///////////////////////////////////////////////////////////////////////////////////////////
/// \class CTCPInterface
/// Physical interface driver for a single TCP connection
///////////////////////////////////////////////////////////////////////////////////////////
class CTCPInterface
{
friend class CTCPLayer;
public:

   CTCPInterface( const char* szIntfName, 
                  unsigned long ulIPAddr, 
                  unsigned short usTCPPort, 
                  unsigned long ulConnectTimeout,
                  unsigned long ulSendTimeout,
                  unsigned long ulResetTimeout,
                  unsigned long ulKeepAliveTimeout);

   ~CTCPInterface();

  long                  Start                ( PFN_NETXCON_DEVICE_RECEIVE_CALLBACK pfnReceiveCallback, 
                                               void*                               pvUser);
  void                  Stop                 ( void);

  long                  Send                 ( unsigned char* pabData,  
                                               unsigned long  ulDataLen);

  long                  GetShortInterfaceName ( unsigned long ulSize, char* szName);
  long                  GetLongInterfaceName  ( unsigned long ulSize, char* szName);

  //////////////////////////////////////////////////
  /// Get state of the interface
  ///   \return State of the interface (see NETX_INTERFACE_STATE_E)
  //////////////////////////////////////////////////
  unsigned long         GetInterfaceState    ( void) { return m_ulState; }

  //////////////////////////////////////////////////
  /// Get keep alive timeout of the interface
  ///   \return Keep alive timeout in ms
  //////////////////////////////////////////////////
  unsigned long         GetKeepAliveTimeout  ( void) { return m_ulKeepAliveTimeout;}  

  //////////////////////////////////////////////////
  /// Get reset timeout of the interface
  ///   \return Reset timeout in ms
  //////////////////////////////////////////////////
  unsigned long         GetResetTimeout      ( void) { return m_ulResetTimeout;}  

  //////////////////////////////////////////////////
  /// Get send timeout of the interface
  ///   \return Send timeout in ms
  //////////////////////////////////////////////////
  unsigned long         GetSendTimeout       ( void) { return m_ulSendTimeout;}  

  //////////////////////////////////////////////////
  /// Get IP address of the interface (Network byte order)
  ///   \return IP address of the interface (Network byte order)
  //////////////////////////////////////////////////
  unsigned long         GetIPAddress         ( void) { return m_tSockAddr.sin_addr.s_addr;}

  //////////////////////////////////////////////////
  /// Get TCP port of the interface (Network byte order)
  ///   \return TCP port of the interface (Network byte order)
  //////////////////////////////////////////////////
  unsigned short        GetTCPPort           ( void) { return m_tSockAddr.sin_port;}

  //////////////////////////////////////////////////
  /// Set keep alive timeout of the interface
  ///   \param ulKeepAliveTimeout Keep alive timeout in ms
  //////////////////////////////////////////////////
  void                  SetKeepAliveTimeout  ( unsigned long ulKeepAliveTimeout) { m_ulKeepAliveTimeout = ulKeepAliveTimeout;}  

  //////////////////////////////////////////////////
  /// Set reset timeout of the interface
  ///   \param ulResetTimeout Reset timeout in ms
  //////////////////////////////////////////////////
  void                  SetResetTimeout      ( unsigned long ulResetTimeout) { m_ulResetTimeout = ulResetTimeout;}  

  //////////////////////////////////////////////////
  /// Set send timeout of the interface
  ///   \param ulSendTimeout Send timeout in ms
  //////////////////////////////////////////////////
  void                  SetSendTimeout       ( unsigned long ulSendTimeout) { m_ulSendTimeout = ulSendTimeout;}  

  //////////////////////////////////////////////////
  /// Set IP address of the interface (Network byte order)
  ///   \param ulIPAddress IP address of the interface (Network byte order)
  //////////////////////////////////////////////////
  void                  SetIPAddress         ( unsigned long ulIPAddress) { m_tSockAddr.sin_addr.s_addr = ulIPAddress;}

  //////////////////////////////////////////////////
  /// Set TCP port of the interface (Network byte order)
  ///   \param usTCPPort TCP port of the interface (Network byte order)
  //////////////////////////////////////////////////
  void                  SetTCPPort           ( unsigned short usTCPPort) { m_tSockAddr.sin_port = usTCPPort;}

protected:
  std::string                         m_szInterfaceName;   /*!< Name of the Interface                          */
  
  HANDLE                              m_hCommThread;       /*!< Handle of Communication Thread                 */
  HANDLE                              m_hCommThreadStop;   /*!< Stop Communication Thread Event Handle         */
  
  HANDLE                              m_hWSAEvent;         /*!< Socket event handler                           */  
  HANDLE                              m_hSendEvent;        /*!< Closed connection event handler                */  
  
  SOCKET                              m_hSocket;           /*!< Socket handle                                  */ 
  SOCKADDR_IN                         m_tSockAddr;         /*!< Socket address                                 */
  unsigned long                       m_ulState;           /*!< Interface state                                */
  
  unsigned long                       m_ulConnectTimeout;   /*!< Scan Timeout                                  */
  unsigned long                       m_ulResetTimeout;     /*!< Reset     Timeout                             */
  unsigned long                       m_ulSendTimeout;      /*!< Send      Timeout                             */
  unsigned long                       m_ulKeepAliveTimeout; /*!< KeepAlive Timeout                             */

  PFN_NETXCON_DEVICE_RECEIVE_CALLBACK m_pfnReceiveCallback;/*!< Receive-Callback ( Function-Callback))         */
  void*                               m_pvReceiveUserData; /*!< Received data                                  */

  CRITICAL_SECTION                    m_tcsInterfaceLock;

  static DWORD WINAPI   CommThread     ( void* pvParam);
  DWORD                 CommThreadFunc ( void);

  SOCKET                CreateSocket   ( void);

  long                  OpenConnection ( SOCKET        hSocket, 
                                         SOCKADDR_IN*  ptSockAddr, 
                                         unsigned long ulConnectTimeout);

  unsigned long         QueryHalfOpenOutboundTCPConnections ( void);

};

/*****************************************************************************/
/*! \}                                                                       */
/*****************************************************************************/


